﻿using System;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.Localization;
using UnityEngine.Localization.Components;
using TMPro;

namespace PK
{
	[ExecuteAlways]
    public class DialogWindow : MonoBehaviour
    {
		[SerializeField] private GameObject _characterPanel;
		[SerializeField] private GameObject _backgroundPanel;
		[SerializeField] private GameObject _bottomPanel;
		[SerializeField] private GameObject _backgroundTitlePanel;
		[SerializeField] private GameObject _messageSkip;

		[Space]
		[SerializeField] private AnimatedImage _portraitImage;
		[SerializeField] private AnimatedImage _backgroundImage;
		[SerializeField] private LocalizeStringEvent _characterTitleLabel;
		[SerializeField] private LocalizeStringEvent _backgroundTitleLabel;
		[SerializeField] private LocalizeStringEvent _characterMessageLabel;
		[SerializeField] private LocalizeStringEvent _messageLabel;

		[Space]
		[SerializeField] private RectTransform _choicesPanel;
		[SerializeField] private RectTransform _choicesParent;
		[SerializeField] private DialogWindowChoice _choicePrefab;

		[SerializeField] private LocalizedString _continueString;

		private int _messageSkipIndex = -1;

		private Action<int> _callback;

		public static DialogWindow Create(Transform parent, IDialogMessageForView messageForView, IDialogChoicesForView choicesForView, Action<int> callback)
		{
			GameObject prefab = Resources.Load("Windows/DialogWindow") as GameObject;
			GameObject gameObject = Instantiate(prefab, parent.transform);
			DialogWindow window = gameObject.GetComponent<DialogWindow>();
			window.Initialize(messageForView, choicesForView, callback);
			return window;
		}
		
		private void Initialize(IDialogMessageForView messageForView, IDialogChoicesForView choicesForView, Action<int> callback)
        {
			_callback = callback;
			HexCharacter character = messageForView == null ? choicesForView.Character : messageForView.Character;
			HexBackground background = messageForView == null ? choicesForView.Background : messageForView.Background;
			LocalizedString message = messageForView == null ? null : (messageForView.Message.IsEmpty ? null : messageForView.Message);

			// Character
			if (character != null)
            {
				_characterPanel.SetActive(true);
				if (message != null)
                {
					_characterMessageLabel.StringReference = message;
					_characterMessageLabel.RefreshString();
				}
				else
                {
					_characterPanel.SetActive(false);
					Debug.LogError("Character message is empty.");
                }
				HexCharacter.AnimationData data = character.GetAnimation(CharacterEmotion.Default);
				if (data != null)
                {
					_portraitImage.Initialize(data.Sprites, data.FramesPerSecond);
				}
			}
			else
            {
				_characterPanel.SetActive(false);
			}

			// Background
			if (background != null)
            {
				_backgroundPanel.SetActive(true);
				HexBackground.AnimationData data = background.Animation;
				if (data != null)
                {
					_backgroundImage.Initialize(data.Sprites, data.FramesPerSecond);
				}
			}
			else
            {
				_backgroundPanel.SetActive(false);
			}

			// Title
			if (background != null)
            {
				if (!background.Name.IsEmpty)
                {
					_backgroundTitlePanel.SetActive(true);
					_backgroundTitleLabel.StringReference = background.Name;
					_backgroundTitleLabel.RefreshString();
				}
				else
                {
					_backgroundTitlePanel.SetActive(false);
					Debug.LogError("Background title is empty.");
				}
			}
			else
            {
				_backgroundTitlePanel.SetActive(false);
			}
			if (character != null)
            {
				if (!character.Name.IsEmpty)
                {
					_characterTitleLabel.StringReference = character.Name;
					_characterTitleLabel.RefreshString();
				}
				else
                {
					Debug.LogError("Character title is empty.");
				}
			}

			// Message
			if (message != null)
            {
				if (character != null)
                {
					_messageLabel.gameObject.SetActive(false);
					_characterMessageLabel.StringReference = message;
					_characterMessageLabel.RefreshString();
				}
				else
                {
					_messageLabel.StringReference = message;
					_messageLabel.RefreshString();
				}
			}
			else
            {
				_messageLabel.gameObject.SetActive(false);
			}

			// Choices panel always visible at least for Continue button
			_choicesPanel.gameObject.SetActive(true);

			if (choicesForView != null)
			{
				_messageSkip.gameObject.SetActive(false);
				for (int i = 0; i < choicesForView.Choices.Count; i++)
				{
					ChoiceLineData choiceData = choicesForView.Choices[i];
					DialogWindowChoice choice = Instantiate(_choicePrefab, _choicesParent);
					choice.Initialize(i, choiceData.Line, OnChoiceClick, choiceData.WasChoosed);
				}
			}
			else
			{
				_messageSkip.gameObject.SetActive(true);
				_messageSkipIndex = -1;
				DialogWindowChoice choice = Instantiate(_choicePrefab, _choicesParent);
				choice.Initialize(0, _continueString, OnChoiceClick, false, true);
			}
        }

		public void OnSkipMessageClick()
        {
			_callback?.Invoke(_messageSkipIndex);
			Destroy(gameObject);
		}

		private void OnChoiceClick(int index)
        {
			_callback?.Invoke(index);
			Destroy(gameObject);
		}

		//private void UpdateBackground()
  //      {
		//	Canvas.ForceUpdateCanvases();

		//	Vector3[] messageCorners = new Vector3[4];
		//	Vector3[] choicesCorners = new Vector3[4];
		//	_messagePanel.GetWorldCorners(messageCorners);
		//	_choicesPanel.GetWorldCorners(choicesCorners);

		//	Vector3 minBound = Vector3.Min(messageCorners[0], choicesCorners[0]);
		//	Vector3 maxBound = Vector3.Max(messageCorners[2], choicesCorners[2]);

		//	RectTransform parent = GetComponent<RectTransform>();
		//	Vector3 localMin = parent.InverseTransformPoint(minBound);
		//	Vector3 localMax = parent.InverseTransformPoint(maxBound);

		//	Vector2 size = localMax - localMin;
		//	Vector2 center = (localMin + localMax) / 2f;

		//	_background.sizeDelta = size;
		//	_background.anchoredPosition = center;
		//}

		/// <summary>
		/// Used purely for link handling in publisher DEMO
		/// </summary>
		private void Update()
		{
			// TODO: Remove link handling when we no longer need it for DEMO
			if(false == Input.GetMouseButtonDown(0))
			{
				return;
			}
			TMP_Text textMessage = _characterMessageLabel.GetComponent<TMP_Text>();
			int linkIndex = TMPro.TMP_TextUtilities.FindIntersectingLink(textMessage, Input.mousePosition, Camera.main);
			if (linkIndex == -1)
			{
				return;
			}
			TMP_LinkInfo linkInfo = textMessage.textInfo.linkInfo[linkIndex];
			string selectedLink = linkInfo.GetLinkID();
			if (selectedLink != "") {
				Debug.LogFormat("Open link {0}", selectedLink);
				Application.OpenURL(selectedLink);
			}
		}
	}
}
