﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK.Tactical
{
    public class TacticalAIController : AIController
    {
        private Savefile.MapData _mapData;
        private Queue<HexCreatureSquadModel> _squadQueue = new();

        public TacticalAIController(Savefile.MapData mapData, Player player) : base(player)
        {
            _mapData = mapData;
        }

        public override void PrepareTaskQueue()
        {
            IEnumerable<HexCreatureSquadModel> squads = _mapData.Map.GetEnumerable<HexCreatureSquadModel>().Where((s) => s.Player == Player && s.ParentSquadId == 0);
            IEnumerable<HexCreatureSquadModel> targetSquads = _mapData.Map.GetEnumerable<HexCreatureSquadModel>().Where((s) => s.Player != Player);

            List<(HexCreatureSquadModel squad, int distance)> targets = new();
            foreach (HexCreatureSquadModel squad in squads)
            {
                Vector2Int axialPosition = HexHelper.OddToAxial(squad.Position);
                targets.Add(targetSquads.Select((s) => (squad, HexHelper.AxialDistance(HexHelper.OddToAxial(s.Position), axialPosition))).OrderBy((s) => s.Item2).FirstOrDefault());
            }
            foreach ((HexCreatureSquadModel squad, int distance) in targets.OrderBy((s) => s.distance))
            {
                _squadQueue.Enqueue(squad);
            }
        }

        public override AITask GetNextTask()
        {
            if (_squadQueue.Count == 0)
            {
                return null;
            }
            HexCreatureSquadModel squad = _squadQueue.Dequeue();

            IEnumerable<HexCreatureSquadModel> squads = _mapData.Map.GetEnumerable<HexCreatureSquadModel>().Where((s) => s.Player == Player);
            IEnumerable<HexCreatureSquadModel> targetSquads = _mapData.Map.GetEnumerable<HexCreatureSquadModel>().Where((s) => s.Player != Player);

            Vector2Int position = PathfindingHelper.FindEnemy(squad.Position, new HashSet<Vector2Int>(targetSquads.Select((s) => s.Position)), _mapData.ObstacleMask, _mapData.Map.Size, new HashSet<Vector2Int>(squads.Select((s) => s.Position)));
            if (position != HexHelper.INVALID_POSITION)
            {
                HexCreatureSquadModel targetSquad = targetSquads.FirstOrDefault((s) => s.Position == position);
                if (targetSquad != null)
                {
                    return new AIMoveTowardsAndAttackTask(squad.Id, targetSquad.Id);
                }
            }
            return null;
        }
    }
}
