﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK.Tactical
{
    public partial class GameController
    {
        private class DamageSquadsCommand : Command
        {
            private List<HexCreatureSquadModel> _targetSquads;
            private List<HexCreatureSquadModel> _allSquads;
            private float _damage;

            public DamageSquadsCommand(List<HexCreatureSquadModel> targetSquads, float damage)
            {
                _targetSquads = targetSquads;
                _damage = damage;
            }

            protected override CommandResult ExecuteBeforeChilds(IContext context)
            {
                DamageSquads(context);
                RegroupOrSurrender(context);
                return CommandResult.Complete;
            }

            private void DamageSquads(IContext context)
            {
                if (_targetSquads.Count > 0)
                {
                    _allSquads = new();
                    SquadHelper.GetAllSquads(_targetSquads[0].Id, context.MapData.Map, _allSquads);
                }

                foreach (HexCreatureSquadModel targetSquad in _targetSquads)
                {
                    float dealedDamage = 0;
                    List<HexCreatureSquadModel.UnitData> units = new(targetSquad.Units);
                    while (dealedDamage < _damage)
                    {
                        bool anyAlive = false;
                        ListHelper.Shuffle(units);
                        foreach (HexCreatureSquadModel.UnitData unit in units)
                        {
                            if (unit.Health > 0)
                            {
                                anyAlive = true;
                                float unitDamage = Mathf.Min(unit.Health, Random.Range(0f, 0.25f) * _damage);
                                unit.Health -= unitDamage;
                                dealedDamage += unitDamage;
                            }
                            if (dealedDamage >= _damage)
                            {
                                break;
                            }
                        }
                        if (!anyAlive || dealedDamage >= _damage)
                        {
                            break;
                        }
                    }
                }
                context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadsHit(_targetSquads.Select((s) => s as ICreatureSquadModelForView).ToList(), _allSquads.Select((s) => s as ICreatureSquadModelForView).ToList());
            }

            private void RegroupOrSurrender(IContext context)
            {
                if (SquadHelper.AreSurrendering(_allSquads))
                {
                    List<(ICreatureSquadModelForView, Vector2Int)> surrenderData = new();
                    HexMapModel map = context.MapData.Map;
                    
                    foreach (HexCreatureSquadModel squad in _allSquads)
                    {
                        squad.Enabled = false;
                        Vector2Int leftTarget = squad.Position;
                        Vector2Int rightTarget = squad.Position;
                        while (map.GetTerrainTile(leftTarget.x, leftTarget.y) != 0)
                        {
                            leftTarget += Vector2Int.left;
                        }
                        while (map.GetTerrainTile(rightTarget.x, rightTarget.y) != 0)
                        {
                            rightTarget += Vector2Int.right;
                        }
                        surrenderData.Add((squad, Vector2Int.Distance(leftTarget, squad.Position) < Vector2Int.Distance(rightTarget, squad.Position) ? (leftTarget + Vector2Int.right) : (rightTarget + Vector2Int.left)));
                    }
                    bool anyAlive = _allSquads.Any((s) => s.Units.Any((u) => u.Health > 0));
                    context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadsSurrender(surrenderData);
                    context.MapData.RefreshObstacleMask();
                    context.EventManager.Get<MapDataEvents>().CallOnRefreshObstacleMask();
                }
                else
                {
                    if (_allSquads.Count == 1)
                    {
                        return;
                    }
                    List<int> squadsAliveUnitsCount = new();

                    int minUnitsCount = int.MaxValue;
                    int maxUnitsCount = int.MinValue;
                    foreach (HexCreatureSquadModel squad in _allSquads)
                    {
                        int aliveUnitsCount = 0;
                        foreach (HexCreatureSquadModel.UnitData unit in squad.Units)
                        {
                            if (unit.Health > 0f)
                            {
                                ++aliveUnitsCount;
                            }
                        }
                        minUnitsCount = Mathf.Min(minUnitsCount, aliveUnitsCount);
                        maxUnitsCount = Mathf.Max(maxUnitsCount, aliveUnitsCount);
                        squadsAliveUnitsCount.Add(aliveUnitsCount);
                    }
                    if (maxUnitsCount - minUnitsCount <= 1)
                    {
                        return;
                    }

                    int targetSquadSize = Mathf.FloorToInt(squadsAliveUnitsCount.Sum() / _allSquads.Count);
                    List<(ulong, ulong, int, int)> unitSwaps = new();
                    for (int i = 0; i < _allSquads.Count; i++)
                    {
                        HexCreatureSquadModel toSquad = _allSquads[i];
                        int unitsToGet = targetSquadSize - squadsAliveUnitsCount[i];
                        for (int j = 0; j < unitsToGet; j++)
                        {
                            int toIndex = GetFirstDeadUnitIndex(toSquad);
                            for (int k = 0; k < _allSquads.Count; k++)
                            {
                                if (i == k)
                                {
                                    continue;
                                }
                                HexCreatureSquadModel fromSquad = _allSquads[k];
                                int unitsToGive = squadsAliveUnitsCount[k] - targetSquadSize;
                                if (unitsToGive > 0)
                                {
                                    int fromIndex = GetLastAliveUnitIndex(fromSquad);
                                    unitSwaps.Add((fromSquad.Id, toSquad.Id, fromIndex, toIndex));
                                    HexCreatureSquadModel.UnitData unit = toSquad.Units[toIndex];
                                    toSquad.Units[toIndex] = fromSquad.Units[fromIndex];
                                    fromSquad.Units[fromIndex] = unit;
                                    ++squadsAliveUnitsCount[i];
                                    --squadsAliveUnitsCount[k];
                                    break;
                                }
                            }
                        }
                    }
                    context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadsRegroup(unitSwaps);
                }
            }

            private int GetLastAliveUnitIndex(HexCreatureSquadModel squad)
            {
                for (int i = squad.Units.Count - 1; i >= 0; i--)
                {
                    if (squad.Units[i].Health > 0)
                    {
                        return i;
                    }
                }
                return -1;
            }

            private int GetFirstDeadUnitIndex(HexCreatureSquadModel squad)
            {
                for (int i = 0; i < squad.Units.Count; i++)
                {
                    if (squad.Units[i].Health <= 0)
                    {
                        return i;
                    }
                }
                return -1;
            }
        }
    }
}
