﻿using System.Linq;
using UnityEngine;

namespace PK.Tactical
{
    public partial class GameController
    {
        private class PlaceCreatureSquadsRandomly : Command
        {
            private HexCreatureSquadModel[] _squads;
            private Player _player;

            public PlaceCreatureSquadsRandomly(HexCreatureSquadModel[] squads, Player player)
            {
                _squads = squads;
                _player = player;
            }

            protected override CommandResult ExecuteBeforeChilds(IContext context)
            {
                if (_squads == null || _squads.Length == 0)
                {
                    Debug.LogWarning("Player has no squads.");
                    return CommandResult.Complete;
                }

                HexSpawnAreaModel spawnArea = null;
                foreach (HexSpawnAreaModel area in context.MapData.Map.GetEnumerable<HexSpawnAreaModel>())
                {
                    if (area.Player == _player)
                    {
                        spawnArea = area;
                        break;
                    }
                }
                if (spawnArea != null)
                {
                    for (int i = 0; i < _squads.Length; i++)
                    {
                        HexCreatureSquadModel[] squads = _squads[i].Split(HexDatabase.Instance.GetCreature(_squads[i].Uid).MaxSoldiersPerHex);
                        Vector2Int[] squadPositions = new Vector2Int[squads.Length];

                        int tryCount = 0;
                        while (true)
                        {
                            if (tryCount > 150)
                            {
                                Debug.LogError("Can't place a squad.");
                                break;
                            }

                            bool isPlacementValid = true;
                            Vector2Int position = spawnArea.GetPosition(Random.Range(0, spawnArea.Points.Count));
                            if (!context.MapData.IsEmpty(position))
                            {
                                continue;
                            }
                            Vector2Int axialPosition = HexHelper.OddToAxial(position);

                            for (int j = 0, n = Mathf.Min(6, squadPositions.Length); j < n; j++)
                            {
                                Vector2Int squadPosition = j == 0 ? position : HexHelper.AxialToOdd(axialPosition + HexHelper.AXIAL_DIRECTIONS[j]);
                                if (context.MapData.IsEmpty(squadPosition))
                                {
                                    squadPositions[j] = squadPosition;
                                }
                                else
                                {
                                    isPlacementValid = false;
                                }
                            }
                            if (isPlacementValid)
                            {
                                break;
                            }
                        }

                        for (int j = 0; j < squads.Length; j++)
                        {
                            HexCreatureSquadModel squad = squads[j];
                            squad.ParentSquadId = j > 0 ? squads[0].Id : 0;
                            squad.Position = squadPositions[j];
                            squad.Player = _player;
                            squad.HasAction = true;
                            context.MapData.Map.AddEntity(squad);
                            context.EventManager.Get<MapEvents>().CallOnEntityCreate(squad);
                        }
                        context.MapData.RefreshObstacleMask();
                        context.EventManager.Get<MapDataEvents>().CallOnRefreshObstacleMask();
                        context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadsCreate(squads.Select((s) => s as ICreatureSquadModelForView).ToList());
                    }
                }
                else
                {
                    Debug.LogError("Can't find area to place creatures.");
                }
                return CommandResult.Complete;
            }
        }
    }
}
