﻿using System.Collections.Generic;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

namespace PK
{
    public class CreatureSquadBarIcon : MonoBehaviour
    {
        public enum Result
        {
            Click,
            Enter,
            Exit
        }

        private static int _outlineColorId = Shader.PropertyToID("_OutlineColor");

        [SerializeField] private ButtonExt _button;
        [SerializeField] private Image _iconImage;
        [SerializeField] private Image _healthbarImage;
        [SerializeField] private Image _selectionImage;
        [SerializeField] private TMP_Text _unitCountText;

        [SerializeField] private Material _hoverMaterial;
        [SerializeField] private Material _selectionMaterial;

        private List<ICreatureSquadModelForView> _squads = new();
        private System.Action<Result> _callback;

        private void OnEnable()
        {
            _button.onClick.AddListener(OnClick);
            _button.onHoverBegin.AddListener(OnHoverBegin);
            _button.onHoverEnd.AddListener(OnHoverEnd);
            _selectionImage.enabled = false;
        }

        private void OnDisable()
        {
            _button.onClick.RemoveListener(OnClick);
            _button.onHoverBegin.RemoveListener(OnHoverBegin);
            _button.onHoverEnd.RemoveListener(OnHoverEnd);
        }

        public void Refresh(List<ICreatureSquadModelForView> squads, bool forceActive, System.Action<Result> callback = null)
        {
            _squads.Clear();
            _squads.AddRange(squads);

            HexCreature creature = HexDatabase.Instance.GetCreature(squads[0].Uid);
            _button.interactable = forceActive ? true : squads[0].HasAction;
            if (creature.RoundIcon != null)
            {
                _iconImage.sprite = creature.RoundIcon;
            }
            float health = 0;
            float maxHealth = 0;
            int unitCount = 0;
            foreach (ICreatureSquadModelForView squad in squads)
            {
                health += squad.Health;
                maxHealth += squad.MaxHealth;
                unitCount += squad.AliveUnitCount;
            }
            _healthbarImage.fillAmount = health / maxHealth;
            _unitCountText.text = unitCount.ToString();
            _callback = callback;
        }

        public bool IsSquad(ulong id)
        {
            foreach (ICreatureSquadModelForView squad in _squads)
            {
                if (squad.Id == id)
                {
                    return true;
                }
            }
            return false;
        }

        public void SetSelectionState(SquadSelectionState state)
        {
            switch (state)
            {
                case SquadSelectionState.None:
                    _selectionImage.enabled = false;
                    break;
                case SquadSelectionState.Hovered:
                    _selectionImage.enabled = true;
                    _selectionImage.color = _hoverMaterial.GetColor(_outlineColorId);
                    break;
                case SquadSelectionState.Selected:
                    _selectionImage.enabled = true;
                    _selectionImage.color = _selectionMaterial.GetColor(_outlineColorId);
                    break;
            }
        }

        public void SetActive(bool active)
        {
            gameObject.SetActive(active);
        }

        private void OnClick()
        {
            _callback?.Invoke(Result.Click);
        }

        private void OnHoverBegin()
        {
            _callback?.Invoke(Result.Enter);
        }

        private void OnHoverEnd()
        {
            _callback?.Invoke(Result.Exit);
        }
    }
}
