﻿using UnityEngine;

namespace PK.Tactical
{
    public class FogOfWarView : MonoBehaviour, IEventReceiver
    {
        private static int _mainTexPropertyId = Shader.PropertyToID("_MainTex");

        [SerializeField] private Material _fogMaterial;

        private Renderer _renderer;
        private MeshFilter _meshFilter;
        private Mesh _mesh;
        private Material _material;

        void IEventReceiver.AddEvents()
        {
            Initialize();

            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnUpdateFogOfWar += OnUpdateFogOfWar;
        }

        void IEventReceiver.RemoveEvents()
        {
            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnUpdateFogOfWar -= OnUpdateFogOfWar;
        }

        private void OnUpdateFogOfWar(bool[] mask, Vector2Int size)
        {
            SpriteHelper helper = new();
            HexMask hexMask = HexDatabase.Instance.Mask;
            Sprite maskSprite = hexMask.GetBigGroundSprite();
            for (int i = 0; i < size.x; i++)
            {
                for (int j = 0; j < size.y; j++)
                {
                    Vector2Int position = new Vector2Int(i, j);
                    Vector3 tilePosition = HexHelper.GetTilePosition(position);
                    if (!mask[j * size.x + i])
                    {
                        helper.Add(tilePosition, maskSprite);
                    }
                    else
                    {
                        for (int k = 0; k < 6; k++)
                        {
                            Vector2Int nearbyPos = HexHelper.AxialToOdd(HexHelper.OddToAxial(position) + HexHelper.AXIAL_DIRECTIONS[k]);
                            if (nearbyPos.x < 0 || nearbyPos.y < 0 || nearbyPos.x >= size.x || nearbyPos.y >= size.y)
                            {
                                continue;
                            }
                            if (!mask[nearbyPos.y * size.x + nearbyPos.x])
                            {
                                helper.Add(tilePosition, hexMask.GetSprite((HexDirection)(1 << k)));
                            }
                        }
                    }
                }
            }
            _material.SetTexture(_mainTexPropertyId, maskSprite.texture);
            helper.FillMesh(_mesh);
        }

        private void Initialize()
        {
            if (_renderer == null)
            {
                hideFlags = HideFlags.HideInInspector | HideFlags.DontSave;

                _material = Instantiate(_fogMaterial);

                _renderer = gameObject.AddComponent<MeshRenderer>();
                _renderer.hideFlags = HideFlags.HideInInspector | HideFlags.DontSave;
                _renderer.sharedMaterial = _material;
                _renderer.sortingOrder = SortingOrder.FOG_OF_WAR;

                _meshFilter = gameObject.AddComponent<MeshFilter>();
                _meshFilter.hideFlags = HideFlags.HideInInspector | HideFlags.DontSave;

                _mesh = new Mesh();
                _mesh.hideFlags = HideFlags.DontSave;
                _meshFilter.mesh = _mesh;
            }
        }
    }
}
