﻿using DG.Tweening;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK.Tactical
{
    public partial class TacticalMapView
    {
        private Dictionary<ulong, HexBanerView> _baners = new();

        private void AddCreatureSquadsEvents()
        {
            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnCreatureSquadsCreate += OnCreatureSquadsCreate;
            events.OnCreatureSquadsMove += OnCreatureSquadsMove;
            events.OnCreatureSquadsRegroup += OnCreatureSquadsRegroup;
            events.OnCreatureSquadsStartAttack += OnCreatureSquadsStartAttack;
            events.OnCreatureSquadsAttackMelee += OnCreatureSquadsAttackMelee;
            events.OnCreatureSquadsAttackRanged += OnCreatureSquadsAttackRanged;
            events.OnCreatureSquadsSurrender += OnCreatureSquadsSurrender;
            events.OnCreatureSquadsHit += OnCreatureSquadsHit;
            events.OnCreatureSquadUpdate += OnCreatureSquadUpdate;
            events.OnUpdateFogOfWar += OnUpdateFogOfWar;
        }

        private void RemoveCreatureSquadsEvents()
        {
            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnCreatureSquadsCreate -= OnCreatureSquadsCreate;
            events.OnCreatureSquadsMove -= OnCreatureSquadsMove;
            events.OnCreatureSquadsRegroup -= OnCreatureSquadsRegroup;
            events.OnCreatureSquadsAttackMelee -= OnCreatureSquadsAttackMelee;
            events.OnCreatureSquadsAttackRanged -= OnCreatureSquadsAttackRanged;
            events.OnCreatureSquadsSurrender -= OnCreatureSquadsSurrender;
            events.OnCreatureSquadsHit -= OnCreatureSquadsHit;
            events.OnCreatureSquadUpdate -= OnCreatureSquadUpdate;
            events.OnUpdateFogOfWar -= OnUpdateFogOfWar;
        }

        private void EnableCreatureSquads()
        {
            foreach (var baner in _baners)
            {
                Destroy(baner.Value.gameObject);
            }
            _baners.Clear();
        }

        private void OnCreatureSquadsCreate(List<ICreatureSquadModelForView> squads)
        {
            if (_baners.ContainsKey(squads[0].Id))
            {
                Debug.LogError("Trying to duplicate a baner.");
                return;
            }
            HexCreatureSquadView squadView = (_entities[squads[0].Id] as HexCreatureSquadView);
            HexCreature creature = HexDatabase.Instance.GetCreature(squadView.Uid);
            Player player = squadView.Player;
            HexBanerView banerPrefab = player == TacticalGameMediator.Instance.SelfPlayer ? creature.SelfTacticalBanerPrefab : creature.EnemyTacticalBanerPrefab;
            if (banerPrefab != null)
            {
                HexBanerView baner = Instantiate(banerPrefab, transform, false);
                baner.Initialize(squads.Select((s) => _entities[s.Id] as HexCreatureSquadView));
                _baners.Add(squads[0].Id, baner);
            }
        }

        private void OnCreatureSquadsMove(List<(ICreatureSquadModelForView squad, Vector2Int start, Vector2Int end)> data)
        {
            foreach ((ICreatureSquadModelForView squad, Vector2Int start, Vector2Int end) in data)
            {
                if (_entities.TryGetValue(squad.Id, out HexEntityView view) && view is HexCreatureSquadView squadView)
                {
                    _locker.Lock(LOCK);
                    squadView.Move(start, end, () =>
                    {
                        _locker.Unlock(LOCK);
                    });
                }
                if (_baners.TryGetValue(squad.Id, out HexBanerView baner))
                {
                    if (TacticalGameMediator.Instance.IsVisible(end))
                    {
                        baner.Activate();
                    }
                }
            }
        }

        private void OnCreatureSquadsRegroup(List<(ulong, ulong, int, int)> data)
        {
            foreach ((ulong fromSquadId, ulong toSquadId, int fromUnitIndex, int toUnitIndex) in data)
            {
                if (_entities.TryGetValue(fromSquadId, out HexEntityView fromView) && fromView is HexCreatureSquadView fromSquadView && _entities.TryGetValue(toSquadId, out HexEntityView toView) && toView is HexCreatureSquadView toSquadView)
                {
                    fromSquadView.MoveUnit(fromUnitIndex, toSquadView, toUnitIndex);
                }
            }
        }

        private void OnCreatureSquadsStartAttack(List<ulong> targetIds)
        {
            foreach (ulong id in targetIds)
            {
                if (_baners.TryGetValue(id, out HexBanerView baner))
                {
                    Vector3 targetPosition = baner.transform.position;
                    if (!CameraHelper.IsVisible(targetPosition, _camera))
                    {
                        _locker.Lock(LOCK);
                        MoveCameraTo(targetPosition, () =>
                        {
                            _locker.Unlock(LOCK);
                        });
                        break;
                    }
                }
            }
        }

        private void OnCreatureSquadsAttackMelee(List<ulong> ids, List<ulong> targetIds)
        {
            foreach (ulong id in ids)
            {
                if (_entities.TryGetValue(id, out HexEntityView view) && view is HexCreatureSquadView squadView)
                {
                    bool isLeft = false;
                    foreach (ulong targetId in targetIds)
                    {
                        if (_entities.TryGetValue(targetId, out HexEntityView enemyView) && !Mathf.Approximately(view.Position.x, enemyView.Position.x))
                        {
                            isLeft = enemyView.Position.x < view.Position.x;
                        }
                    }
                    squadView.Attack(isLeft);
                }
            }
        }

        private void OnCreatureSquadsAttackRanged(List<ulong> ids, List<Vector2Int> targets)
        {
            float time = 1.25f;
            HexRangedCreature creature = null;
            Sequence sequnce = DOTween.Sequence();
            List<SpriteRenderer> projectiles = new List<SpriteRenderer>(ids.Count * targets.Count);
            _locker.Lock(LOCK);
            foreach (ulong id in ids)
            {
                if (_entities.TryGetValue(id, out HexEntityView view) && view is HexCreatureSquadView squadView)
                {
                    if (creature == null)
                    {
                        creature = HexDatabase.Instance.GetCreature(view.Uid) as HexRangedCreature;
                    }
                    for (int i = 0; i < targets.Count; i++)
                    {
                        Vector3 start = view.Position;
                        Vector3 end = HexHelper.GetTilePosition(targets[i]) + (Vector3)(Random.insideUnitCircle * 0.25f);
                        Vector3 center = (start + end) / 2.0f + Vector3.up * 3.0f;
                        SpriteRenderer projectileRenderer = SpriteRendererPool.Get();
                        projectileRenderer.sprite = creature.Projectile;
                        projectileRenderer.sortingOrder = SortingOrder.PROJECTILE;
                        projectileRenderer.transform.position = start;
                        projectiles.Add(projectileRenderer);
                        Transform projectileTransform = projectileRenderer.transform;
                        sequnce.Join(DOVirtual.Float(0f, 1f, time, (float value) =>
                        {
                            Vector3 currentPosition = projectileTransform.position;
                            Vector3 nextPosition = BezierHelper.Quadratic(start, center, end, value);
                            projectileTransform.position = nextPosition;
                            projectileTransform.right = (nextPosition - currentPosition).normalized;
                        }).SetEase(Ease.Linear));
                    }
                    bool isLeft = false;
                    foreach (Vector2Int target in targets)
                    {
                        Vector3 targetPosition = HexHelper.GetTilePosition(target);
                        if (!Mathf.Approximately(view.Position.x, targetPosition.x))
                        {
                            isLeft = targetPosition.x < view.Position.x;
                        }
                    }
                    squadView.Attack(isLeft);
                }
            }
            sequnce.OnComplete(() =>
            {
                foreach (SpriteRenderer projectile in projectiles)
                {
                    SpriteRendererPool.Release(projectile);
                }
                _locker.Unlock(LOCK);
            });
        }

        private void OnCreatureSquadsSurrender(List<(ICreatureSquadModelForView, Vector2Int)> data)
        {
            foreach ((ICreatureSquadModelForView squad, Vector2Int target) in data)
            {
                ulong id = squad.Id;
                if (_entities.TryGetValue(id, out HexEntityView view) && view is HexCreatureSquadView squadView)
                {
                    squadView.Surrender(target);
                    _entities.Remove(id);
                }
                if (_baners.TryGetValue(id, out HexBanerView baner))
                {
                    baner.Destroy();
                    _baners.Remove(id);
                }
            }
        }

        private void OnCreatureSquadsHit(List<ICreatureSquadModelForView> targets, List<ICreatureSquadModelForView> allTargets)
        {
            foreach (ICreatureSquadModelForView target in targets)
            {
                if (_entities.TryGetValue(target.Id, out HexEntityView view) && view is HexCreatureSquadView squadView)
                {
                    squadView.Hit(target);
                }
            }
            if (_baners.TryGetValue(allTargets[0].Id, out HexBanerView baner))
            {
                baner.UpdateHealth(allTargets);
            }
        }

        private void OnCreatureSquadUpdate(ulong id, bool hasAction)
        {
            if (_baners.TryGetValue(id, out HexBanerView baner))
            {
                baner.UpdateAction(hasAction);
            }
        }

        private void OnUpdateFogOfWar(bool[] mask, Vector2Int size)
        {
            foreach (HexCreatureSquadModel squad in _map.Model.GetEnumerable<HexCreatureSquadModel>())
            {
                ulong id = squad.ParentSquadId == 0 ? squad.Id : squad.ParentSquadId;
                if (_baners.TryGetValue(id, out HexBanerView baner))
                {
                    if (TacticalGameMediator.Instance.IsVisible(squad.Position))
                    {
                        baner.Activate();
                    }
                }
            }
        }
    }
}
