using UnityEngine;
using UnityEngine.Rendering;
using UnityEngine.Rendering.Universal;

namespace PK
{
    public class ObjectShadowsRendererFeature : ScriptableRendererFeature
    {
        private ObjectShadowsRenderPass _renderPass;

        public override void AddRenderPasses(ScriptableRenderer renderer, ref RenderingData renderingData)
        {
            _renderPass.renderPassEvent = RenderPassEvent.BeforeRenderingOpaques;
            _renderPass.ConfigureInput(ScriptableRenderPassInput.None);

            renderer.EnqueuePass(_renderPass);
        }

        public override void Create()
        {
            _renderPass = new ObjectShadowsRenderPass();
        }

        protected override void Dispose(bool disposing)
        {
            _renderPass.Dispose();
        }
    }

    public class ObjectShadowsRenderPass : ScriptableRenderPass
    {
        private static int _shadowsId = Shader.PropertyToID("_Shadows");

        private FilteringSettings _filteringSettings;
        private RenderStateBlock _renderStateBlock;
        private RTHandle _textureHandle;

        public ObjectShadowsRenderPass()
        {
            _filteringSettings = new FilteringSettings(RenderQueueRange.transparent)
            {
                layerMask = ~0
            };
             _renderStateBlock = new RenderStateBlock(RenderStateMask.Nothing);
        }

        public void Dispose()
        {
            _textureHandle?.Release();
        }

        private void ReAllocate(RenderTextureDescriptor desc)
        {
            desc.msaaSamples = 1;
            desc.graphicsFormat = UnityEngine.Experimental.Rendering.GraphicsFormat.R32G32B32A32_SFloat;
            desc.depthBufferBits = (int)DepthBits.None;
            RenderingUtils.ReAllocateIfNeeded(ref _textureHandle, desc, name: "_Shadows");
        }

        public override void OnCameraSetup(CommandBuffer cmd, ref RenderingData renderingData)
        {
            ResetTarget();
            ReAllocate(renderingData.cameraData.cameraTargetDescriptor);
        }

        public override void Execute(ScriptableRenderContext context, ref RenderingData renderingData)
        {
            string sampleName = "Draw shadows";

            CommandBuffer cmd = CommandBufferPool.Get();
            cmd.BeginSample(sampleName);
            CoreUtils.SetRenderTarget(cmd, _textureHandle, ClearFlag.Color);
            context.ExecuteCommandBuffer(cmd);

            SortingSettings sortingSettings = new SortingSettings(renderingData.cameraData.camera);
            sortingSettings.criteria = SortingCriteria.CommonTransparent;
            sortingSettings.distanceMetric = DistanceMetric.CustomAxis;
            sortingSettings.customAxis = Vector3.up;

            DrawingSettings drawingSettings = new DrawingSettings();
            drawingSettings.SetShaderPassName(0, new ShaderTagId("Shadow2D"));
            drawingSettings.sortingSettings = sortingSettings;

            context.DrawRenderers(renderingData.cullResults, ref drawingSettings, ref _filteringSettings, ref _renderStateBlock);
            
            cmd.Clear();
            cmd.SetGlobalTexture(_shadowsId, _textureHandle);
            cmd.EndSample(sampleName);
            context.ExecuteCommandBuffer(cmd);
            CommandBufferPool.Release(cmd);
        }
    }
}
