﻿using System;
using System.Collections.Generic;
using System.Linq;
using UnityEditor.Searcher;
using UnityEngine;

namespace PK
{
    public class EntitySearchWindowProvider : ScriptableObject
    {
        private Dictionary<ulong, NodeEntry> _nodeEntries = new();
        private List<KeyValuePair<ulong, string>> _names = new();

        public string GetName(ulong id)
        {
            if (_nodeEntries.Count == 0)
            {
                GenerateEntries();
            }

            if (_nodeEntries.TryGetValue(id, out NodeEntry entry))
            {
                return entry.Name;
            }

            return null;
        }

        public IReadOnlyList<KeyValuePair<ulong, string>> GetNames()
        {
            if (_nodeEntries.Count == 0)
            {
                GenerateEntries();
            }

            return _names;
        }

        public void Clear()
        {
            _nodeEntries.Clear();
        }

        public Searcher LoadSearchWindow()
        {
            if (_nodeEntries.Count == 0)
            {
                GenerateEntries();
            }

            List<SearcherItem> root = new();
            NodeEntry dummyEntry = new NodeEntry();

            foreach (NodeEntry nodeEntry in _nodeEntries.Values)
            {
                SearcherItem parent = null;
                string[] entryTitle = nodeEntry.MenuItem.Split('/');

                for (int i = 0; i < entryTitle.Length; i++)
                {
                    var pathEntry = entryTitle[i];
                    var children = parent != null ? parent.Children : root;
                    var item = children.Find(x => x.Name == pathEntry);

                    if (item == null)
                    {
                        if (i == entryTitle.Length - 1)
                        {
                            item = new SearchNodeItem(pathEntry, nodeEntry);
                        }
                        else
                        {
                            item = new SearchNodeItem(pathEntry, dummyEntry);
                        }
                        
                        if (parent != null)
                        {
                            parent.AddChild(item);
                        }
                        else
                        {
                            children.Add(item);
                        }
                    }

                    parent = item;

                    if (parent.Depth == 0 && !root.Contains(parent))
                    {
                        root.Add(parent);
                    }
                }
            }
            SearcherDatabase nodeDatabase = SearcherDatabase.Create(root, string.Empty, false);
            return new Searcher(nodeDatabase, new SearchWindowAdapter("Select Entity"));
        }

        public bool OnSelectEntry(SearcherItem selectedEntry, Action<ulong> callback)
        {
            if (selectedEntry == null)
            {
                return false;
            }
            callback?.Invoke((selectedEntry as SearchNodeItem).NodeEntry.Id);
            return true;
        }

        private void GenerateEntries()
        {
            _nodeEntries.Clear();
            _names.Clear();
            HexMapEditorWindow view = HexMapEditorWindow.Instance;
            if (view == null)
            {
                return;
            }
            IEnumerable<HexEntityModel> entities = view.Map.Model.GetEnumerable(true).Where((m) => !string.IsNullOrEmpty(m.Name));
            foreach (IGrouping<string, HexEntityModel> group in entities.GroupBy((e) => e.Name).Where((g) => g.Count() > 1))
            {
                Debug.LogError($"Name \"{group.Key}\" is used multiple times.");
            }
            foreach (HexEntityModel entity in entities.OrderBy((e) => e.Name))
            {
                _nodeEntries.Add(entity.Id, new NodeEntry(entity.Id, entity.Name, entity.Name));
                _names.Add(new KeyValuePair<ulong, string>(entity.Id, entity.Name));
            }
        }

        public class SearchWindowAdapter : SearcherAdapter
        {
            public override bool HasDetailsPanel => false;

            public SearchWindowAdapter(string title) : base(title)
            {
            }
        }

        public class SearchNodeItem : SearcherItem
        {
            private NodeEntry _nodeEntry;

            public NodeEntry NodeEntry { get { return _nodeEntry; } }

            public SearchNodeItem(string name, NodeEntry nodeEntry) : base(name)
            {
                _nodeEntry = nodeEntry;
            }
        }

        public struct NodeEntry : IEquatable<NodeEntry>
        {
            private ulong _id;
            private string _name;
            private string _menuItem;

            public ulong Id { get { return _id; } }
            public string Name { get { return _name; } }
            public string MenuItem { get { return _menuItem; } }

            public NodeEntry(ulong id, string name, string menuItem)
            {
                _id = id;
                _name = name;
                _menuItem = menuItem;
            }

            public bool Equals(NodeEntry other)
            {
                return Equals(_menuItem, other._menuItem) && _id == other._id;
            }
        }
    }
}
