﻿using System;
using System.Collections.Generic;
using UnityEditor;
using UnityEditor.Searcher;
using UnityEngine;
using UnityEngine.UIElements;
using static PK.HexVariablesModel;

namespace PK
{
    public class EntityButton : VisualElement
    {
        private SerializedProperty _property;

        public EntityButton(BaseGraphEditorNode node, SerializedProperty property)
        {
            _property = property;
            LabelButton entityIdButton = null;
            entityIdButton = new LabelButton("Entity", GetText(), () =>
            {
                MapDataHelper.SelectEntity(node.View.Window, (id) =>
                {
                    _property.ulongValue = id;
                    _property.serializedObject.ApplyModifiedProperties();
                    entityIdButton.Button.text = GetText();
                });
            });
            Add(entityIdButton);
        }

        public void Update(bool visible)
        {
            this.visible = visible;
        }

        private string GetText()
        {
            return MapDataHelper.GetEntityName(_property.ulongValue) ?? "None";
        }
    }

    public class VariableButton : VisualElement
    {
        private SerializedProperty _property;

        public VariableButton(BaseGraphEditorNode node, SerializedProperty property)
        {
            _property = property;
            LabelButton idButton = null;
            idButton = new LabelButton("Variable", GetText(), () =>
            {
                MapDataHelper.SelectVariable(node.View.Window, (id) =>
                {
                    property.ulongValue = id;
                    property.serializedObject.ApplyModifiedProperties();
                    idButton.Button.text = GetText();
                });
            });
            Add(idButton);
        }

        private string GetText()
        {
            return MapDataHelper.GetVariable(_property.ulongValue)?.Name ?? "None";
        }
    }

    public static class MapDataHelper
    {
        private static EntitySearchWindowProvider _entitySearchProvider;
        private static VariableSearchWindowProvider _variableSearchProvider;

        static MapDataHelper()
        {
            _entitySearchProvider = ScriptableObject.CreateInstance<EntitySearchWindowProvider>();
            _variableSearchProvider = ScriptableObject.CreateInstance<VariableSearchWindowProvider>();
        }

        public static void SelectEntity(EditorWindow window, Action<ulong> callback)
        {
            SearcherWindow.Show(window, _entitySearchProvider.LoadSearchWindow(),
                   item => _entitySearchProvider.OnSelectEntry(item, callback),
                  GUIUtility.GUIToScreenPoint(Event.current.mousePosition) - window.position.position, null);
        }

        public static void SelectVariable(EditorWindow window, Action<ulong> callback)
        {
            SearcherWindow.Show(window, _variableSearchProvider.LoadSearchWindow(),
                    item => _variableSearchProvider.OnSelectEntry(item, callback),
                   GUIUtility.GUIToScreenPoint(Event.current.mousePosition) - window.position.position, null);
        }

        public static void Clear()
        {
            _entitySearchProvider.Clear();
            _variableSearchProvider.Clear();
        }

        public static string GetEntityName(ulong id)
        {
            return _entitySearchProvider.GetName(id);
        }

        public static Variable GetVariable(ulong id)
        {
            return _variableSearchProvider.GetVariable(id);
        }

        public static Variable GetVariable(string name)
        {
            return _variableSearchProvider.GetVariable(name);
        }

        public static IReadOnlyList<KeyValuePair<ulong, string>> GetEntitiesNames()
        {
            return _entitySearchProvider.GetNames();
        }
    }
}
