﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using UnityEngine.Serialization;

namespace PK
{
    [System.Serializable]
    public class HexMapModel : ISerializationCallbackReceiver
    {
        [SerializeField, HideInInspector] private int _serializedVersion;
        [SerializeField] private ulong[] _terrain;
        [SerializeField] private short[] _roads; // first byte contains type, second bytes contains variant index 
        [SerializeField] private int _width;
        [SerializeField] private int _height;

        [SerializeField, FormerlySerializedAs("_props")] private List<HexObjectModel> _objects = new();
        [SerializeField] private List<HexCreatureModel> _creatures = new();
        [SerializeField] private List<HexHeroModel> _heroes = new();

        [SerializeReference, FormerlySerializedAs("_objects")] private List<HexEntityModel> _entities = new();
        [SerializeField] private List<ReferenceKeyValue<HexEntityModel, BaseInteraction>> _interactions = new();
        [SerializeField] private HexVariablesModel _globalVariables;
        [SerializeField] private List<HexGlobalTriggerModel> _globalTriggers = new();

        private Dictionary<ulong, HexEntityModel> _entitiesDict = new();
        private Dictionary<HexEntityModel, BaseInteraction> _interactionsDict = new();

        private int[] _interactionMask; // 1 byte will contain type, 3 bytes will contain custom info (for example VISITABLE tile directions)
        private ulong[] _interactableIdMask;

        [SerializeField] private ulong[] _terrainOverrides;

        public int Width { get { return _width; } }
        public int Height { get { return _height; } }
        public Vector2Int Size { get { return new Vector2Int(_width, _height); } }

        public short[] Roads { get { return _roads; } }

        public HexVariablesModel GlobalVariables { get { return _globalVariables; } }
        public IReadOnlyList<HexGlobalTriggerModel> GlobalTriggers { get { return _globalTriggers; } }

        public int[] InteractionMask { get { return _interactionMask; } }

        public IEnumerable<HexEntityModel> GetEnumerable(bool includeDisabled = false)
        {
            return _entitiesDict.Where((p) => p.Value.Enabled || includeDisabled).Select((p) => p.Value);
        }

        public IEnumerable<T> GetEnumerable<T>(bool includeDisabled = false)
        {
            return _entitiesDict.Where((p) => p.Value.Enabled || includeDisabled).Select((p) => p.Value).OfType<T>();
        }

#if UNITY_EDITOR
        public int GetObjectIndex(ulong id)
        {
            for (int i = 0; i < _entities.Count; i++)
            {
                if (_entities[i].Id == id)
                {
                    return i;
                }
            }
            return -1;
        }
#endif

        public void Validate()
        {
            if (_terrain == null || _terrain.Length == 0)
            {
                _terrain = new ulong[_width * _height];
            }
            if (_roads == null || _roads.Length == 0)
            {
                _roads = new short[_width * _height];
            }
            RefreshInteractionMask();
            // TODO remove empty space
        }

        public void Initialize()
        {
            foreach (HexObjectModel objectModel in GetEnumerable<HexObjectModel>(true))
            {
                HexObject @object = HexDatabase.Instance.GetObject(objectModel.Uid);
                if (@object != null)
                {
                    BaseInteraction interaction = @object.Interaction;
                    if (interaction != null)
                    {
                        objectModel.Interaction = interaction;
                    }
                }
                else
                {
                    Debug.LogError($"Object {@object.Uid} is missing.");
                    continue;
                }
                
            }
            foreach (ReferenceKeyValue<HexEntityModel, BaseInteraction> pair in _interactions)
            {
                if (pair.Key != null && pair.Key is HexInteractableEntityModel interactableEntity)
                {
                    interactableEntity.Interaction = pair.Value;
                }
            }
        }

        public void RefreshInteractionMask()
        {
            if (_interactionMask == null || _interactionMask.Length != _terrain.Length)
            {
                _interactionMask = new int[_terrain.Length];
                _interactableIdMask = new ulong[_terrain.Length];
            }
            if (_terrainOverrides == null || _terrainOverrides.Length != _terrain.Length)
            {
                _terrainOverrides = new ulong[_terrain.Length];
            }

            System.Array.Fill(_interactionMask, 0);
            System.Array.Fill(_interactableIdMask, 0u);
            foreach (HexObjectModel objectModel in GetEnumerable<HexObjectModel>())
            {
                ulong objectUid = objectModel.Uid;
                HexObject @object = HexDatabase.Instance.GetObject(objectUid);
                if (@object != null)
                {
                    foreach (KeyValue<Vector2Int, int> pair in @object.InteractionMask)
                    {
                        Vector2Int position = new Vector2Int(pair.Key.x + objectModel.Position.x, pair.Key.y + objectModel.Position.y);
                        if (position.x < 0 || position.y < 0 || position.x >= _width || position.y >= _height)
                        {
                            continue;
                        }
                        int index = position.y * _width + position.x;
                        int value = pair.Value;
                        HexObjectMask maskType = HexObjectMaskHelper.GetType(value);
                        if ((maskType == HexObjectMask.Visitable || maskType == HexObjectMask.BlockVisitable) && HexObjectMaskHelper.IsTileDataEmpty(value))
                        {
                            value = HexObjectMaskHelper.GetDefaultTileData(value);
                        }
                        _interactionMask[index] = value;
                        _interactableIdMask[index] = objectModel.Id;
                    }
                }
                else
                {
                    Debug.LogError($"Object {objectUid} is missing.");
                }
            }
            // Creatures use block visitable
            foreach (HexCreatureModel creature in GetEnumerable<HexCreatureModel>())
            {
                int index = creature.Position.y * _width + creature.Position.x;
                _interactionMask[index] = HexObjectMaskHelper.GetDefaultTileData((int)HexObjectMask.BlockVisitable);
            }
        }

        public ulong GetTerrainTile(int index)
        {
            if (_terrainOverrides[index] != 0)
            {
                return _terrainOverrides[index];
            }
            return _terrain[index];
        }

        public ulong GetTerrainTile(int x, int y)
        {
            if (_terrainOverrides[y * _width + x] != 0)
            {
                return _terrainOverrides[y * _width + x];
            }
            return _terrain[y * _width + x];
        }

        public ulong GetTerrainTile(Vector2Int position)
        {
            if (_terrainOverrides[position.y * _width + position.x] != 0)
            {
                return _terrainOverrides[position.y * _width + position.x];
            }
            return _terrain[position.y * _width + position.x];
        }


        public int GetInteractionMask(int index)
        {
            return _interactionMask[index];
        }

        public int GetInteractionMask(int x, int y)
        {
            return _interactionMask[y * _width + x];
        }

        public ulong GetInteractableId(int index)
        {
            return _interactableIdMask[index];
        }

        public ulong GetInteractableId(int x, int y)
        {
            return _interactableIdMask[y * _width + x];
        }

        public (byte type, byte index) GetRoad(Vector2Int position)
        {
            return GetRoad(position.y * _width + position.x);
        }

        public (byte type, byte index) GetRoad(int index)
        {
            short road = _roads[index];
            return ((byte)(road & 0xFF), (byte)(road >> 8));
        }

        public bool HasRoad(Vector2Int position)
        {
            return HasRoad(position.y * _width + position.x);
        }

        public bool HasRoad(int index)
        {
            short road = _roads[index];
            if (road == 0)
            {
                return false;
            }
            return true;
        }

        public bool CanReplaceTerrainTile(Vector2Int position, ulong uid)
        {
            if (position.x < 0 || position.y < 0)
            {
                return false;
            }
            if (position.x >= _width || position.y >= _height)
            {
                return true;
            }
            return GetTerrainTile(position) != uid;
        }

        public void ReplaceTerrainTile(Vector2Int position, ulong uid)
        {
            if (position.x < 0 || position.y < 0)
            {
                return;
            }
            // Resize if needed
            if (position.x >= _width || position.y >= _height)
            {
                ResizeMap(Mathf.Max(position.x + 1, _width), Mathf.Max(position.y + 1, _height));
            }
            _terrain[position.y * _width + position.x] = uid;
        }

        public void SetTileOverride(Vector2Int position, ulong tileUid)
        {
            if (tileUid == _terrain[position.y * _width + position.x])
            {
                _terrainOverrides[position.y * _width + position.x] = 0;
            }
            else
            {
                _terrainOverrides[position.y * _width + position.x] = tileUid;
            }
        }

        public void ResetTileOverride(Vector2Int position)
        {
            _terrainOverrides[position.y * _width + position.x] = 0;
        }

        public void ReplaceRoad(Vector2Int position, byte type, byte variant)
        {
            if (position.x < 0 || position.y < 0)
            {
                return;
            }
            // Resize if needed
            if (position.x >= _width || position.y >= _height)
            {
                ResizeMap(Mathf.Max(position.x + 1, _width), Mathf.Max(position.y + 1, _height));
            }
            _roads[position.y * _width + position.x] = (short)(type | (variant << 8));
        }

        public void MoveTerrain(Vector2Int direction)
        {
            //Vector2Int newSize = HexHelper.AxialToOdd(HexHelper.OddToAxial(new Vector2Int(_width - 1, _height - 1)) + direction) + Vector2Int.one;
            Vector2Int newSize = new Vector2Int(_width + direction.x, _height + direction.y);
            ulong[] newTerrain = new ulong[newSize.x * newSize.y];
            short[] newRoads = new short[newSize.x * newSize.y];
            for (int i = 0; i < _width; i++)
            {
                for (int j = 0; j < _height; j++)
                {
                    //Vector2Int newPosition = HexHelper.AxialToOdd(HexHelper.OddToAxial(new Vector2Int(i, j)) + direction);
                    Vector2Int newPosition = new Vector2Int(i, j) + direction;
                    if (newPosition.x >= 0 && newPosition.x <= newSize.x && newPosition.y >= 0 && newPosition.y <= newSize.y)
                    {
                        newTerrain[newPosition.y * newSize.x + newPosition.x] = _terrain[j * _width + i];
                        newRoads[newPosition.y * newSize.x + newPosition.x] = _roads[j * _width + i];
                    }
                }
            }
            _terrain = newTerrain;
            _roads = newRoads;
            _width = newSize.x;
            _height = newSize.y;
        }

        public HexEntityModel GetEntity(ulong id)
        {
            if (_entitiesDict.TryGetValue(id, out HexEntityModel entity))
            {
                return entity;
            }
            return null;
        }

        public bool CanPlaceObject(Vector2Int position)
        {
            if (position.x < 0 || position.y < 0)
            {
                return false;
            }

            return true;
        }

        public ulong PlaceObject(Vector2Int position, Vector2 offset, ulong uid)
        {
            ulong id = GenerateId();
            HexObjectModel @object = new HexObjectModel(uid, id, position, offset);
            _entitiesDict[id] = @object;
            _entities.Add(@object);
            RefreshInteractionMask();
            return id;
        }

        public ulong AddEntity(HexEntityModel entity)
        {
            ulong id = GenerateId();
            entity.Id = id;
            _entitiesDict[id] = entity;
            _entities.Add(entity);
            RefreshInteractionMask();
            return id;
        }

        public HexObjectModel GetObject(ulong id)
        {
            if (_entitiesDict.TryGetValue(id, out HexEntityModel entity) && entity is HexObjectModel @object)
            {
                return @object;
            }
            return null;
        }

        public bool CanPlaceCreature(Vector2Int position)
        {
            return _entitiesDict.Count((p) => p.Value.Position == position) == 0;
        }

        public ulong PlaceCreature(Vector2Int position, ulong uid)
        {
            ulong id = GenerateId();
            HexCreatureModel creature = new HexCreatureModel(uid, id, position);
            _entitiesDict[id] = creature;
            _entities.Add(creature);
            return id;
        }

        public bool CanPlaceHero(Vector2Int position)
        {
            return _entitiesDict.Count((p) => p.Value.Position == position) == 0;
        }

        public ulong PlaceHero(Vector2Int position, ulong uid)
        {
            ulong id = GenerateId();
            HexHeroModel hero = new HexHeroModel(uid, id, position);
            _entitiesDict[id] = hero;
            _entities.Add(hero);
            return id;
        }

        public ulong PlaceProximityTrigger(Vector2Int position)
        {
            ulong id = GenerateId();
            HexProximityTriggerModel trigger = new HexProximityTriggerModel(id, position);
            _entitiesDict[id] = trigger;
            _entities.Add(trigger);
            return id;
        }

        public ulong PlaceSpawnPoint(Vector2Int position)
        {
            ulong id = GenerateId();
            HexSpawnAreaModel spawnPoint = new HexSpawnAreaModel(id, position);
            _entitiesDict[id] = spawnPoint;
            _entities.Add(spawnPoint);
            return id;
        }

        public ulong PlaceMapZone(Vector2Int position)
        {
            ulong id = GenerateId();
            HexMapZoneModel mapZone = new HexMapZoneModel(id, position);
            _entitiesDict[id] = mapZone;
            _entities.Add(mapZone);
            return id;
        }

        public void MoveEntity(Vector2Int position, Vector2 offset, ulong id)
        {
            if (_entitiesDict.TryGetValue(id, out HexEntityModel entity))
            {
                entity.Position = position;
                if (entity is HexObjectModel @object)
                {
                    @object.Offset = offset;
                    RefreshInteractionMask();
                }
            }
        }

        public void RemoveEntity(ulong id)
        {
            if (_entitiesDict.TryGetValue(id, out HexEntityModel entity))
            {
                _entitiesDict.Remove(id);
                _entities.Remove(entity);
                if (entity is HexObjectModel)
                {
                    RefreshInteractionMask();
                }
            }
        }

        public HexHeroModel GetHero(ulong id)
        {
            if (_entitiesDict.TryGetValue(id, out HexEntityModel @object) && @object is HexHeroModel heroObject)
            {
                return heroObject;
            }
            return null;
        }

        private void ResizeMap(int newWidth, int newHeight)
        {
            ulong[] newTerrain = new ulong[newWidth * newHeight];
            short[] newRoads = new short[newWidth * newHeight];
            for (int i = 0; i < _width; i++)
            {
                for (int j = 0; j < _height; j++)
                {
                    newTerrain[j * newWidth + i] = _terrain[j * _width + i];
                    newRoads[j * newWidth + i] = _roads[j * _width + i];
                }
            }
            _terrain = newTerrain;
            _roads = newRoads;
            _width = newWidth;
            _height = newHeight;
            RefreshInteractionMask();
        }

        public bool IsInBounds(Vector2Int position)
        {
            return position.x >= 0 && position.y >= 0 && position.x < _width && position.y < _height;
        }

        public Vector2Int Clamp(Vector2Int position)
        {
            position.x = Mathf.Clamp(position.x, 0, _width - 1);
            position.y = Mathf.Clamp(position.y, 0, _height - 1);
            return position;
        }

        public bool IsInteractable(ulong id)
        {
            return _entitiesDict[id] is HexInteractableEntityModel;
        }

        public bool HasInteraction(ulong id)
        {
            return _interactionsDict.ContainsKey(_entitiesDict[id]);
        }

        public BaseInteraction GetInteraction(ulong id)
        {
            return _interactionsDict[_entitiesDict[id]];
        }

        public void AddInteraction(ulong id)
        {
            HexEntityModel @object = _entitiesDict[id];
            _interactionsDict[@object] = null;
            _interactions.Add(new ReferenceKeyValue<HexEntityModel, BaseInteraction>(@object, null));
        }

#if UNITY_EDITOR
        public void AddInteraction(BaseInteraction interaction, ulong id)
        {
            HexEntityModel @object = _entitiesDict[id];
            _interactionsDict[@object] = interaction;
            _interactions.Add(new ReferenceKeyValue<HexEntityModel, BaseInteraction>(@object, interaction));
        }
#endif

        public void RemoveInteraction(ulong id)
        {
            HexEntityModel @object = _entitiesDict[id];
            if (_interactionsDict.ContainsKey(@object))
            {
                _interactionsDict.Remove(@object);
                _interactions.RemoveAll((p) => p.Key == @object);
            }
        }

#if UNITY_EDITOR

        public int GetInteractionIndex(ulong id)
        {
            for (int i = 0; i < _interactions.Count; i++)
            {
                if (_interactions[i].Key.Id == id)
                {
                    return i;
                }
            }
            return -1;
        }

#endif

        void ISerializationCallbackReceiver.OnBeforeSerialize()
        {
        }

        void ISerializationCallbackReceiver.OnAfterDeserialize()
        {
            if (_serializedVersion < 1)
            {
                if (_entities.Count == 0)
                {
                    foreach (HexObjectModel @object in _objects)
                    {
                        _entitiesDict[@object.Id] = @object;
                        _entities.Add(@object);
                    }
                    foreach (HexCreatureModel creature in _creatures)
                    {
                        _entitiesDict[creature.Id] = creature;
                        _entities.Add(creature);
                    }
                    foreach (HexHeroModel hero in _heroes)
                    {
                        _entitiesDict[hero.Id] = hero;
                        _entities.Add(hero);
                    }
                }
                _objects.Clear();
                _creatures.Clear();
                _heroes.Clear();
                _serializedVersion = 1;
            }
            if (_serializedVersion < 2)
            {
                foreach (ReferenceKeyValue<HexEntityModel, BaseInteraction> interaction in _interactions)
                {
                    if (interaction.Value is ActionInteraction actionInteraction)
                    {
                        foreach (ActionModifyVariableNode node in actionInteraction.Graph.Nodes.OfType<ActionModifyVariableNode>())
                        {
                            node.Id = _globalVariables.AddVariableIfNotExists(node.Name, node.Type);
                        }
                        foreach (ActionCheckVariableNode node in actionInteraction.Graph.Nodes.OfType<ActionCheckVariableNode>())
                        {
                            node.Id = _globalVariables.AddVariableIfNotExists(node.Name, node.Type);
                        }
                    }
                }
                foreach (HexGlobalTriggerModel trigger in _globalTriggers)
                {
                    foreach (ActionModifyVariableNode node in trigger.Graph.Nodes.OfType<ActionModifyVariableNode>())
                    {
                        node.Id = _globalVariables.AddVariableIfNotExists(node.Name, node.Type);
                    }
                    foreach (ActionCheckVariableNode node in trigger.Graph.Nodes.OfType<ActionCheckVariableNode>())
                    {
                        node.Id = _globalVariables.AddVariableIfNotExists(node.Name, node.Type);
                    }
                }
                _serializedVersion = 2;
            }

            _entitiesDict.Clear();
            foreach (HexEntityModel entity in _entities)
            {
                if (entity.Id == 0)
                {
                    entity.Id = GenerateId();
                }
                _entitiesDict[entity.Id] = entity;
            }
            _interactionsDict.Clear();
            foreach (ReferenceKeyValue<HexEntityModel, BaseInteraction> pair in _interactions)
            {
                if (pair.Key != null)
                {
                    _interactionsDict[pair.Key] = pair.Value;
                }
            }
        }

        private ulong GenerateId()
        {
            ulong id;
            do
            {
                id = HexHelper.GenerateUID();
            }
            while (_entitiesDict.ContainsKey(id));
            return id;
        }
    }
}
