using System.Linq;
using UnityEngine;
using UnityEngine.Rendering;

public static class IconRenderer
{
    private static Material _material;
    private static Mesh _mesh;

    public static void Render(Sprite sprite, Texture2D output, Vector2 scale)
    {
        if (sprite == null)
        {
            return;
        }

        RenderTexture renderTexture = RenderTexture.GetTemporary(output.width, output.height);

        if (_mesh == null)
        {
            _mesh = new Mesh();
            _mesh.MarkDynamic();
        }
        else
        {
            _mesh.Clear();
        }

        Vector2[] vertices = sprite.vertices;
        float centerY = 0;
        foreach (Vector2 vertex in vertices)
        {
            centerY += vertex.y / vertices.Length;
        }

        _mesh.vertices = vertices.Select((v) => new Vector3(v.x, v.y - centerY)).ToArray();
        _mesh.uv = sprite.uv;
        _mesh.triangles = sprite.triangles.Select((t) => (int)t).ToArray();
        _mesh.UploadMeshData(false);
        _mesh.RecalculateBounds();

        if (_material == null)
        {
            Shader shader = Shader.Find("Universal Render Pipeline/2D/Sprite-Unlit-Default");
            _material = new Material(shader);
        }
        _material.SetTexture("_MainTex", sprite.texture);

        Vector2 size = _mesh.bounds.size * scale;

        CommandBuffer cmd = CommandBufferPool.Get();
        cmd.SetRenderTarget(renderTexture, 0, CubemapFace.Unknown, 0);
        cmd.ClearRenderTarget(true, true, Color.clear);
        cmd.SetViewProjectionMatrices(Matrix4x4.identity, Matrix4x4.Ortho(-size.x, size.x, -size.y, size.y, 0.001f, 1000f));
        cmd.DrawMesh(_mesh, Matrix4x4.TRS(new Vector3(0, 0, -10), Quaternion.identity, Vector3.one), _material, 0, 0);
        Graphics.ExecuteCommandBuffer(cmd);
        CommandBufferPool.Release(cmd);

        RenderTexture active = RenderTexture.active;
        RenderTexture.active = renderTexture;
        output.ReadPixels(new Rect(0, 0, output.width, output.height), 0, 0);
        output.Apply();
        RenderTexture.active = active;

        RenderTexture.ReleaseTemporary(renderTexture);
    }
}
