using System.Linq;
using UnityEditor;
using UnityEngine;
using UnityEngine.UIElements;

namespace PK
{
    public class HexObjectEditorWindow : PreviewSceneView
    {
        private static class Styles
        {
            public static readonly GUIContent panelTitle = EditorGUIUtility.TrTextContent("Settings");
        }

        private enum Tool
        {
            Mask,
            Offset,
            Shadow
        }

        private enum TileEditMode
        {
            None,
            Enter,
            Exit
        }

        private static string[] TOOLS_TEXTS = new string[]
        {
            "Mask",
            "Offset",
            "Shadow"
        };

        public static Color[] MASK_COLORS = new Color[]
        {
            Color.clear,
            new Color(1, 0, 0, 0.15f),
            new Color(0, 1, 0, 0.15f),
            new Color(1, 1, 0, 0.15f),
            new Color(0, 0, 1, 0.15f)
        };

        private static Color[] TILE_EDIT_COLORS = new Color[]
        {
            Color.clear,
            new Color(1, 0, 0, 1),
            new Color(0, 1, 0, 1)
        };

        private static Vector2Int INVALID_SELECTION = new Vector2Int(-1000, 0);

        private static Texture2D _backgroundTexture;
        private static Sprite _backgroundSprite;

        [SerializeField] private HexObject _object;

        private SerializedObject _serializedObject;
        private HexObjectView _view;

        private Tool _selectedTool;
        private Vector2Int _selectedTile = INVALID_SELECTION;
        private TileEditMode _tileEditMode;

        private VisualElement _settingsInspectorVisualElement;

        private SerializedProperty _isAligned;
        private SerializedProperty _interactionMask;
        private SerializedProperty _offset;
        private SerializedProperty _shadowOffset;

        [MenuItem("PK/Object Editor")]
        public static void Open()
        {
            Open(null);
        }

        public static void Open(HexObject @object)
        {
            HexObjectEditorWindow window = (HexObjectEditorWindow)GetWindow(typeof(HexObjectEditorWindow), false, "Object Editor");
            window.Show();
            window.Initialize(@object);
        }

        [UnityEditor.Callbacks.OnOpenAsset(1)]
        public static bool OpenAsset(int instanceID, int line)
        {
            Object obj = EditorUtility.InstanceIDToObject(instanceID);
            if (obj is HexObject @object)
            {
                Open(@object);
                return true;
            }
            return false;
        }

        protected override void OnEnable()
        {
            base.OnEnable();

            GameObject backgroundObject = new GameObject();
            backgroundObject.transform.localScale = new Vector3(100f, 100f, 1f);
            SpriteRenderer backgroundRenderer = backgroundObject.AddComponent<SpriteRenderer>();
            backgroundRenderer.sharedMaterial = HexMaterials.Instance.TileMaterial;
            backgroundRenderer.sortingOrder = -1;
            backgroundRenderer.sprite = GetBackgroundSprite();
            Add(backgroundObject);

            GameObject viewObject = new GameObject();
            _view = viewObject.AddComponent<HexObjectView>();
            Add(viewObject);

            if (_settingsInspectorVisualElement == null)
            {
                _settingsInspectorVisualElement = CreateSettingsInspectorVisualElement();
                rootVisualElement.Add(_settingsInspectorVisualElement);
            }

            if (_object != null)
            {
                Initialize(_object);
            }
            
            Undo.undoRedoPerformed += OnUndoRedoPerformed;
        }

        protected override void OnDisable()
        {
            base.OnDisable();

            if (_serializedObject != null)
            {
                _serializedObject.ApplyModifiedProperties();
            }

            Undo.undoRedoPerformed -= OnUndoRedoPerformed;
        }

        private void OnUndoRedoPerformed()
        {
            if (_object == null)
            {
                return;
            }

            UpdateView();
        }

        private void Initialize(HexObject @object)
        {
            _object = @object;
            if (_serializedObject != null)
            {
                _serializedObject.Dispose();
            }
            _serializedObject = new SerializedObject(_object);
            _isAligned = _serializedObject.FindProperty("_isAligned");
            _interactionMask = _serializedObject.FindProperty("_interactionMask");
            _offset = _serializedObject.FindProperty("_offset");
            _shadowOffset = _serializedObject.FindProperty("_shadowOffset");
            UpdateView();
        }

        private void OnSettingsGUI()
        {
            if (_object == null)
            {
                return;
            }

            _serializedObject.Update();

            GUILayout.BeginArea(new Rect(0, 0, 256, 256), Styles.panelTitle, GUI.skin.window);

            Tool newSelectedTool = (Tool)(GUILayout.Toolbar((int)_selectedTool, TOOLS_TEXTS));
            if (newSelectedTool != _selectedTool)
            {
                _selectedTool = newSelectedTool;
                Repaint();
            }

            EditorGUI.BeginChangeCheck();
            switch (_selectedTool)
            {
                case Tool.Mask:
                    EditorGUILayout.HelpBox("Click left button to edit tile.\nClick right button to rotate mask type.", MessageType.Info);
                    if (_selectedTile != INVALID_SELECTION)
                    {
                        string modeName = _tileEditMode == TileEditMode.Enter ? "entrance" : "exit";
                        EditorGUILayout.HelpBox($"Is editing tile {modeName}.", MessageType.Info);
                    }
                    break;
                case Tool.Offset:
                    EditorGUILayout.PropertyField(_isAligned);
                    if (_isAligned.boolValue)
                    {
                        EditorGUILayout.PropertyField(_offset);
                    }
                    break;
                case Tool.Shadow:
                    EditorGUILayout.PropertyField(_shadowOffset);
                    break;
            }
            if (EditorGUI.EndChangeCheck())
            {
                _serializedObject.ApplyModifiedProperties();
                UpdateView();
                hasUnsavedChanges = true;
            }

            GUILayout.EndArea();
        }

        protected override void OnDrawHandles()
        {
            if (_object == null)
            {
                return;
            }

            Vector2Int currentTile = HexHelper.GetHexPosition(GetMouseWorldPosition());
            DrawTileHandle(currentTile, Color.white);

            switch (_selectedTool)
            {
                case Tool.Mask:
                    if (Event.current.type == EventType.MouseDown)
                    {
                        if (Event.current.button == 0)
                        {
                            SwitchTileEditMode(currentTile);
                            Event.current.Use();
                        }
                        else if (Event.current.button == 1)
                        {
                            if (_selectedTile == INVALID_SELECTION)
                            {
                                SwitchMaskType(currentTile);
                            }
                            else
                            {
                                ModifySelectedTile(currentTile);
                            }
                            Event.current.Use();
                        }
                    }
                    break;
                case Tool.Offset:
                    if (_object.IsAligned)
                    {
                        EditorGUI.BeginChangeCheck();
                        Vector3 position = _view.Position;
                        Vector3 newPosition = Handles.PositionHandle(position, Quaternion.identity);
                        if (position != newPosition)
                        {
                            _offset.vector2Value = HexHelper.GetOffset(HexHelper.GetTilePosition(Vector2Int.zero), newPosition);
                        }
                        if (EditorGUI.EndChangeCheck())
                        {
                            _serializedObject.ApplyModifiedProperties();
                            UpdateView();
                            hasUnsavedChanges = true;
                        }
                    }
                    DrawTileHandle(Vector2Int.zero, Color.white);
                    break;
            }

            foreach (KeyValue<Vector2Int, int> pair in _object.InteractionMask)
            {
                if (pair.Value != 0)
                {
                    DrawTileHandleFilled(pair.Key, MASK_COLORS[(int)HexObjectMaskHelper.GetType(pair.Value)]);
                }
            }

            if (_selectedTile != INVALID_SELECTION)
            {
                KeyValue<Vector2Int, int> pair = _object.InteractionMask.FirstOrDefault((e) => e.Key == _selectedTile);
                if (pair != null)
                {
                    bool[] tilesData = HexObjectMaskHelper.GetTilesData(pair.Value, _tileEditMode == TileEditMode.Enter);
                    for (int i = 0; i < 6; i++)
                    {
                        Vector2Int position = HexHelper.AxialToOdd(HexHelper.OddToAxial(_selectedTile) + HexHelper.AXIAL_DIRECTIONS[i]);
                        if (tilesData[i])
                        {
                            DrawTileHandleFilled(position, MASK_COLORS[4]);
                        }
                    }
                }
                DrawTileHandle(_selectedTile, TILE_EDIT_COLORS[(int)_tileEditMode]);
            }
        }

        protected override void OnDrawGUI()
        {
            if (_object == null)
            {
                _object = (HexObject)EditorGUILayout.ObjectField(_object, typeof(HexObject), false);
                if (_object != null)
                {
                    Initialize(_object);
                }
            }
        }

        private VisualElement CreateSettingsInspectorVisualElement()
        {
            return new IMGUIContainer(OnSettingsGUI)
            {
                style =
                {
                    flexGrow = 0,
                    flexBasis = 1,
                    flexShrink = 0,
                    minWidth = 256,
                    minHeight = 256,
                    bottom = 5,
                    right = 5,
                    position = new StyleEnum<Position>(Position.Absolute)
                },
                name = "SettingsInspector"
            };
        }

        private void UpdateView()
        {
            _view.Initialize(new HexObjectModel(_object.Uid, 1, Vector2Int.zero, Vector2.zero));
            _view.Refresh(null);
        }

        private Vector3 GetMouseWorldPosition()
        {
            return HandleUtility.GUIPointToWorldRay(Event.current.mousePosition).origin;
        }

        private void DrawTileHandle(Vector2Int position, Color color)
        {
            using (new Handles.DrawingScope(color))
            {
                Handles.DrawPolyLine(HexHelper.GetTileCorners(HexHelper.GetTilePosition(position)));
            }
        }

        private void DrawTileHandleFilled(Vector2Int position, Color color)
        {
            using (new Handles.DrawingScope(color))
            {
                Handles.DrawAAConvexPolygon(HexHelper.GetTileCorners(HexHelper.GetTilePosition(position)));
            }
        }

        private void SwitchMaskType(Vector2Int position)
        {
            int index = System.Array.FindIndex(_object.InteractionMask, (e) => e.Key == position);
            if (index == -1)
            {
                index = _interactionMask.arraySize;
                _interactionMask.InsertArrayElementAtIndex(index);
                SerializedProperty element = _interactionMask.GetArrayElementAtIndex(index);
                element.FindPropertyRelative("_key").vector2IntValue = position;
                element.FindPropertyRelative("_value").intValue = 1;
            }
            else
            {
                SerializedProperty element = _interactionMask.GetArrayElementAtIndex(index);
                SerializedProperty value = element.FindPropertyRelative("_value");
                int nextValue = HexObjectMaskHelper.SwitchMaskModeToNext(value.intValue);
                if (nextValue != 0)
                {
                    value.intValue = nextValue;
                }
                else
                {
                    _interactionMask.DeleteArrayElementAtIndex(index);
                }
            }

            _serializedObject.ApplyModifiedProperties();
        }

        private void ModifySelectedTile(Vector2Int position)
        {
            int index = System.Array.FindIndex(_object.InteractionMask, (e) => e.Key == _selectedTile);
            KeyValue<Vector2Int, int> pair = _object.InteractionMask[index];
            Vector2Int selectedAxial = HexHelper.OddToAxial(_selectedTile);
            bool[] tilesData = HexObjectMaskHelper.GetTilesData(pair.Value, _tileEditMode == TileEditMode.Enter);
            for (int i = 0; i < 6; i++)
            {
                Vector2Int currentAxial = HexHelper.OddToAxial(position);
                if (currentAxial - selectedAxial == HexHelper.AXIAL_DIRECTIONS[i])
                {
                    int newMask = HexObjectMaskHelper.ModifyTilesData(i, pair.Value, _tileEditMode == TileEditMode.Enter, !tilesData[i]);
                    SerializedProperty element = _interactionMask.GetArrayElementAtIndex(index);
                    element.FindPropertyRelative("_value").intValue = newMask;
                    _serializedObject.ApplyModifiedProperties();
                }
            }
        }

        private void SwitchTileEditMode(Vector2Int position)
        {
            int index = System.Array.FindIndex(_object.InteractionMask, (e) => e.Key == position);
            KeyValue<Vector2Int, int> pair = null;
            if (index != -1)
            {
                pair = _object.InteractionMask[index];
            }
            if (_selectedTile == INVALID_SELECTION)
            {
                if (pair != null)
                {
                    switch (HexObjectMaskHelper.GetType(pair.Value))
                    {
                        case HexObjectMask.Visitable:
                        case HexObjectMask.BlockVisitable:
                            _selectedTile = position;
                            _tileEditMode = TileEditMode.Enter;
                            if (HexObjectMaskHelper.IsTileDataEmpty(pair.Value))
                            {
                                SerializedProperty element = _interactionMask.GetArrayElementAtIndex(index);
                                element.FindPropertyRelative("_value").intValue = HexObjectMaskHelper.GetDefaultTileData(pair.Value);
                                _serializedObject.ApplyModifiedProperties();
                            }
                            break;
                    }
                }
            }
            else if (_tileEditMode < TileEditMode.Exit && _selectedTile == position)
            {
                switch (HexObjectMaskHelper.GetType(pair.Value))
                {
                    case HexObjectMask.Visitable:
                        _tileEditMode += 1;
                        break;
                    case HexObjectMask.BlockVisitable:
                        // Blockvisitable has no exit mode
                        _selectedTile = INVALID_SELECTION;
                        break;
                }
            }
            else
            {
                _selectedTile = INVALID_SELECTION;
            }
        }

        private static Sprite GetBackgroundSprite()
        {
            if (_backgroundSprite == null)
            {
                Color color = PreviewSceneView.BACKGROUND_COLOR;
                _backgroundTexture = new Texture2D(4, 4);
                Color[] colors = new Color[16];
                System.Array.Fill(colors, color);
                _backgroundTexture.SetPixels(colors);
                _backgroundTexture.Apply();

                _backgroundSprite = Sprite.Create(_backgroundTexture, new Rect(0, 0, 2, 2), Vector2.one * 0.5f, 0.5f);
            }
            return _backgroundSprite;
        }
    }
}
