﻿using System.Collections.Generic;
using System.Linq;
using UnityEditor;
using UnityEngine;

namespace PK
{
    public class PaintTerrainTool : MapPaintingTool
    {
        private static string PAINT_TERRAIN_BRUSH_SIZE = "PaintTerrainBrushSize";
        private static string PAINT_TERRAIN_BRUSH_DENSITY = "PaintTerrainBrushDensity";

        protected override int BrushSize
        {
            get
            {
                return EditorPrefs.GetInt(PAINT_TERRAIN_BRUSH_SIZE);
            }
            set
            {
                EditorPrefs.SetInt(PAINT_TERRAIN_BRUSH_SIZE, value);
            }
        }

        protected override int BrushDensity
        {
            get
            {
                return EditorPrefs.GetInt(PAINT_TERRAIN_BRUSH_DENSITY);
            }
            set
            {
                EditorPrefs.SetInt(PAINT_TERRAIN_BRUSH_DENSITY, value);
            }
        }

        protected override Color PaintingToolGridColor => Color.white;

        private HexTile _brushSelectedTile;

        private HexTerrain _brushSelectedTerrain;

        private Dictionary<HexTile, int> _terrainTilesWeight = new();

        public override bool IsToolReady()
        {
            return _brushSelectedTile != null || _brushSelectedTerrain != null;
        }

        public override void OnInspectorPanel()
        {
            PaintingToolInspectorPanel();

            bool clickedAny = false;
            foreach (HexTerrain terrain in HexDatabase.Instance.Terrains)
            {
                GUILayout.Label(terrain.name);
                using (new EditorGUI.DisabledScope(terrain.Equals(_brushSelectedTerrain)))
                {
                    if (GUILayout.Button("Use random tiles of " + terrain.name))
                    {
                        _brushSelectedTile = null;
                        _brushSelectedTerrain = terrain;
                        int defaultWeight = 100 / terrain.Tiles.Length;
                        _terrainTilesWeight = terrain.Tiles.ToDictionary(key => key, key => defaultWeight);
                        clickedAny = true;
                    }
                }
                if (_brushSelectedTerrain != null && terrain.Equals(_brushSelectedTerrain))
                {
                    NormalizeSliders();

                    var newValues = _terrainTilesWeight.ToDictionary(kvp => kvp.Key, kvp =>
                    {
                        GUILayout.BeginHorizontal();
                        if (GUILayout.Button(new GUIContent(_editor.GetIcon(kvp.Key.Sprite), kvp.Key.name), GUILayout.Width(BUTTON_SIZE.x), GUILayout.Height(BUTTON_SIZE.y)))
                        {
                            _brushSelectedTile = kvp.Key;
                            _brushSelectedTerrain = null;
                        }
                        int newValue = Mathf.Max(0, EditorGUILayout.IntSlider(kvp.Key.name, kvp.Value, 0, 100));
                        GUILayout.EndHorizontal();
                        return newValue;
                    });
                    if (GUI.changed) DistributeChange(_terrainTilesWeight, newValues);
                }
                else
                {
                    HexTile[] tiles = terrain.Tiles;
                    int i = 0;
                    while (i < tiles.Length)
                    {
                        GUILayout.BeginHorizontal();
                        for (int j = 0, n = (int)(EditorGUIUtility.currentViewWidth / (BUTTON_SIZE.x + 5)); j < n && i < tiles.Length; j++)
                        {
                            HexTile tile = tiles[i];

                            bool isSelected = _brushSelectedTile == tile;
                            if (isSelected)
                            {
                                EditorGUI.BeginDisabledGroup(true);
                            }
                            if (GUILayout.Button(new GUIContent(_editor.GetIcon(tile.Sprite), tile.name), GUILayout.Width(BUTTON_SIZE.x), GUILayout.Height(BUTTON_SIZE.y)))
                            {
                                _brushSelectedTile = tile;
                                _brushSelectedTerrain = null;
                                clickedAny = true;
                            }
                            if (isSelected)
                            {
                                EditorGUI.EndDisabledGroup();
                            }
                            ++i;
                        }
                        GUILayout.EndHorizontal();
                    }
                }
            }

            if (!clickedAny && Event.current.type == EventType.MouseDown)
            {
                 _brushSelectedTerrain = null;
                _brushSelectedTile = null;
                _editor.Repaint();
            }

            if (GUILayout.Button("Move up"))
            {
                RecordUndo();
                _view.DisplaceMap(Vector2Int.up * 2);
            }
            if (GUILayout.Button("Move down"))
            {
                RecordUndo();
                _view.DisplaceMap(Vector2Int.down * 2);
            }
            if (GUILayout.Button("Move left"))
            {
                RecordUndo();
                _view.DisplaceMap(Vector2Int.left);
            }
            if (GUILayout.Button("Move right"))
            {
                RecordUndo();
                _view.DisplaceMap(Vector2Int.right);
            }
        }

        public override void OnMouseMove()
        {
            _hoveredTile = _view.GetHexPosition(GetMouseWorldPosition());
        }

        public override void OnMouseDown()
        {
            if (Event.current.button != 0 || IsToolReady() == false)
            {
                return;
            }
            RecordUndo();
            _view.PauseEditorUpdates();
            List<Vector2Int> neighbours = GetPaintedHexes();
            foreach (Vector2Int neighbour in neighbours)
            {
                HexTile newTile;
                if (_brushSelectedTile != null)
                {
                    newTile = _brushSelectedTile;
                }
                else
                {
                    newTile = GetRandomTerrainTile();
                }

                if (_view.CanReplaceTile(neighbour, newTile))
                {
                    _view.PlaceTile(neighbour, newTile);
                }
            }
            _view.ResumeEditorUpdates();
            Event.current.Use();
        }


        // START: LLM generated code

        private void DistributeChange(Dictionary<HexTile, int> oldValues, Dictionary<HexTile, int> newValues)
        {
            int totalNew = 0;
            foreach (var key in newValues.Keys)
            {
                totalNew += newValues[key];
            }

            if (totalNew != 100)
            {
                int difference = 100 - totalNew;
                int remainingSliders = 0;
                foreach (var key in _terrainTilesWeight.Keys)
                {
                    if (newValues[key] == oldValues[key] && newValues[key] != 0) // Modified condition
                    {
                        remainingSliders++;
                    }
                }

                if (remainingSliders > 0)
                {
                    int changePerSlider = difference / remainingSliders;
                    int remainder = difference % remainingSliders;
                    foreach (var key in _terrainTilesWeight.Keys)
                    {
                        if (newValues[key] == oldValues[key] && newValues[key] != 0) // Modified condition
                        {
                            newValues[key] += changePerSlider;
                            if (remainder > 0)
                            {
                                newValues[key]++;
                                remainder--;
                            }
                        }
                    }
                }
                _terrainTilesWeight = new Dictionary<HexTile, int>(newValues);
            }
            else
            {
                _terrainTilesWeight = new Dictionary<HexTile, int>(newValues);
            }
        }

        private void NormalizeSliders()
        {
            int total = 0;
            foreach (var key in _terrainTilesWeight.Keys)
            {
                total += _terrainTilesWeight[key];
            }

            if (total == 0) return;

            Dictionary<HexTile, int> normalizedValues = new Dictionary<HexTile, int>();

            foreach (var key in _terrainTilesWeight.Keys)
            {
                normalizedValues[key] = Mathf.RoundToInt((float)_terrainTilesWeight[key] / total * 100f);
            }

            _terrainTilesWeight = normalizedValues;

            int normalizedTotal = 0;
            foreach (var key in _terrainTilesWeight.Keys)
            {
                normalizedTotal += _terrainTilesWeight[key];
            }

            if (normalizedTotal != 100)
            {
                int diff = 100 - normalizedTotal;
                HexTile lastKey = new();
                foreach (var key in _terrainTilesWeight.Keys)
                {
                    lastKey = key;
                }
                if (lastKey != null)
                {
                    _terrainTilesWeight[lastKey] += diff;
                }
            }
        }

        public HexTile GetRandomTerrainTile()
        {
            int totalPercentage = _terrainTilesWeight.Sum(kvp => kvp.Value);
            if (totalPercentage != 100)
            {
                Debug.LogWarning("Total percentage is not 100. Adjusting to 100.");
                NormalizeSliders();
            }
            totalPercentage = _terrainTilesWeight.Sum(kvp => kvp.Value);

            int randomValue = Random.Range(0, totalPercentage);
            int cumulativePercentage = 0;

            foreach (var kvp in _terrainTilesWeight)
            {
                cumulativePercentage += kvp.Value;
                if (randomValue < cumulativePercentage)
                {
                    return kvp.Key;
                }
            }

            return _terrainTilesWeight.Keys.Last();
        }

        // END: LLM generated code
    }
}
