﻿using System;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Animations;

namespace PK
{
    public class HexBanerView : MonoBehaviour
    {
        [SerializeField] private SpriteRenderer _banerRenderer;
        [SerializeField] private ParentConstraint _constraint;
        [SerializeField] private Transform _healthbarTransform;
        [SerializeField] private Material _attackableEnemyMaterial;
        [SerializeField] private Material _hoverMaterial;
        [SerializeField] private Material _hoverEnemyMaterial;
        [SerializeField] private Material _selectionMaterial;

        private List<(SpriteRenderer renderer, Color originalColor, Color modifiedColor)> _spritesData = new();
        private Material _defaultMaterial;
        private bool _isActive = false;

        private bool _isHovered = false;
        private bool _isClicked = false;

        private void Awake()
        {
            Color ModifyColor(Color original)
            {
                return new Color(original.r * 0.5f, original.g * 0.5f, original.b * 0.5f, original.a);
            }

            foreach (SpriteRenderer renderer in GetComponentsInChildren<SpriteRenderer>())
            {
                renderer.enabled = false;
                renderer.sortingOrder += SortingOrder.BANER;
                _spritesData.Add((renderer, renderer.color, ModifyColor(renderer.color)));
            }
            _defaultMaterial = _banerRenderer.sharedMaterial;
        }

        public void Initialize(IEnumerable<HexCreatureSquadView> squads)
        {
            foreach (HexCreatureSquadView squad in squads)
            {
                _constraint.AddSource(new ConstraintSource() { sourceTransform = squad.transform, weight = 1 });
            }
            _constraint.constraintActive = true;
        }

        public void SetSelectionState(SquadSelectionState state)
        {
            switch (state)
            {
                case SquadSelectionState.None:
                    _banerRenderer.sharedMaterial = _defaultMaterial;
                    break;
                case SquadSelectionState.AttackableEnemy:
                    _banerRenderer.sharedMaterial = _attackableEnemyMaterial;
                    break;
                case SquadSelectionState.Hovered:
                    _banerRenderer.sharedMaterial = _hoverMaterial;
                    break;
                case SquadSelectionState.HoveredEnemy:
                    _banerRenderer.sharedMaterial = _hoverEnemyMaterial;
                    break;
                case SquadSelectionState.Selected:
                    _banerRenderer.sharedMaterial = _selectionMaterial;
                    break;
            }
        }

        public void UpdateAction(bool hasAction)
        {
            foreach ((SpriteRenderer renderer, Color originalColor, Color modifiedColor) in _spritesData)
            {
                renderer.color = hasAction ? originalColor : modifiedColor;
            }
        }

        public void Activate()
        {
            if (_isActive)
            {
                return;
            }
            _isActive = true;
            foreach ((SpriteRenderer renderer, Color originalColor, Color modifiedColor) data in _spritesData)
            {
                data.renderer.enabled = true;
            }
        }

        public void UpdateHealth(List<ICreatureSquadModelForView> squads)
        {
            float health = 0;
            float maxHealth = 0;
            foreach (ICreatureSquadModelForView squad in squads)
            {
                health += squad.Health;
                maxHealth += squad.MaxHealth;
            }
            _healthbarTransform.localScale = new Vector3(Mathf.Clamp01(health / maxHealth), 1, 1);
        }

        public void Destroy()
        {
            if (gameObject != null)
            {
                Destroy(gameObject);
            }
        }

        public bool IsHovered()
        {
            return _isHovered;
        }

        public bool IsClicked()
        {
            if (_isClicked)
            {
                _isClicked = false;
                return true;
            }
            return false;
        }

        public void OnMouseEnter()
        {
            if (_isActive)
            {
                _isHovered = true;
            }
        }

        public void OnMouseExit()
        {
            if (_isActive)
            {
                _isHovered = false;
            }
        }

        public void OnMouseDown()
        {
            if (_isActive)
            {
                _isHovered = true;
            }
        }

        public void OnMouseUp()
        {
            if (_isActive)
            {
                //_isClicked = false;
            }
        }
    }
}
