﻿using DG.Tweening;
using System.Collections.Generic;
using UnityEngine;

namespace PK
{
    public enum SquadSelectionState
    {
        None,
        Enemy,
        AttackableEnemy,
        Hovered,
        HoveredEnemy,
        Selected,
    }

    public class HexCreatureSquadView : HexEntityView
    {
        const float WALKING_TIME = 0.5f / 1.25f;

        [SerializeField] private Vector3[] _unitPositions;

        private Transform _transform;
        private HexCreatureSquadUnitView _unitViewPrefab;
        private List<HexCreatureSquadUnitView> _units = new();
        private Player _player;

        public override Bounds Bounds { get { return default; } }
        public override Vector3 Position { get { return _transform.position; } }
        public HexCreatureSquadUnitView UnitViewPrefab { set { _unitViewPrefab = value; } }
        public Player Player { get { return _player; } }

        public override void Initialize(IEntityModelForView modelForView)
        {
            _uid = modelForView.Uid;
            _id = modelForView.Id;

            ICreatureSquadModelForView squadModelForView = modelForView as ICreatureSquadModelForView;
            if (_transform == null)
            {
                _transform = transform;
                _player = squadModelForView.Player;

                int unitCount = Mathf.Min(squadModelForView.Units.Count, _unitPositions.Length);
                for (int i = 0; i < unitCount; i++)
                {
                    HexCreatureSquadUnitView unit = Instantiate(_unitViewPrefab, transform, false);
                    unit.transform.localPosition = _unitPositions[i];
                    _units.Add(unit);
                }
            }

            Move(modelForView.Position, Vector2.zero);
        }

        private void OnDestroy()
        {
            DOTween.Kill(_transform);
        }

        public void SetSelectionState(SquadSelectionState state)
        {
            _units.ForEach((u) =>
            {
                if (u != null)
                {
                    u.SetSelectionState(state);
                }
            });
        }

        public override void Move(Vector2Int position, Vector2 offset)
        {
            _transform.position = GetPosition(position);
        }

        public void Move(Vector2Int start, Vector2Int end, System.Action callback)
        {
            Vector3 startPosition = GetPosition(start);
            Vector3 endPosition = GetPosition(end);
            _units.ForEach((u) =>
            {
                if (u != null)
                {
                    u.StartWalking();
                    u.SetFlipX(endPosition.x < startPosition.x);
                }
            });
            _transform.position = startPosition;
            _transform.DOMove(endPosition, WALKING_TIME).SetEase(Ease.Linear).OnComplete(() =>
            {
                _units.ForEach((u) =>
                {
                    if (u != null)
                    {
                        u.EndWalking();
                    }
                });
                callback?.Invoke();
            });
        }

        public void MoveUnit(int index, HexCreatureSquadView other, int otherIndex)
        {
            HexCreatureSquadUnitView unit = _units[index];
            _units[index] = null;
            other._units[otherIndex] = unit;
            unit.transform.SetParent(other._transform, true);
            Vector3 target = other._transform.TransformPoint(other._unitPositions[otherIndex]);
            unit.transform.DOMove(target, WALKING_TIME).SetEase(Ease.Linear);
        }

        public void Attack(bool isLeftDirection)
        {
            _units.ForEach((u) =>
            {
                if (u != null)
                {
                    u.SetFlipX(isLeftDirection);
                    u.Attack();
                }
            });
        }

        public void Hit(ICreatureSquadModelForView modelForView)
        {
            for (int i = 0; i < _units.Count; i++)
            {
                HexCreatureSquadUnitView unit = _units[i];
                if (unit != null)
                {
                    if (modelForView.Units[i].Health <= 0)
                    {
                        _units[i] = null;
                        unit.Kill();
                    }
                    else
                    {
                        unit.Hit();
                    }
                }
            }
        }

        public void Surrender(Vector2Int target)
        {
            Vector3 startPosition = _transform.position;
            Vector3 endPosition = GetPosition(target);
            _units.ForEach((u) =>
            {
                if (u != null)
                {
                    u.StartWalking();
                    u.SetFlipX(endPosition.x < startPosition.x);
                }
            });
            _transform.DOMove(endPosition, WALKING_TIME * Vector3.Distance(_transform.position, endPosition) * 2.0f).SetEase(Ease.Linear).OnComplete(() =>
            {
                Destroy(gameObject);
            });
        }

        private static Vector3 GetPosition(Vector2Int position)
        {
            return HexHelper.GetTilePosition(position) + Vector3.back * 0.1f;
        }
    }
}
